<?php

namespace model\Entity;

use DATABASE\ORM\Interact\Entities\EntityScheme;
use FwHtml\Elements\Tags\Main\HtmlTags;
use FwHtml\Elements\Tags\Table;
use FwHtml\FontAwesome;
use model\Chats;
use model\Orders;
use model\Services;
use model\Users;
use model\UsersTable;

class ChatMessagesEntity extends EntityScheme {
    public $chat_message_id;
    /**
     * محتوا
     **/
    public $message_content;
    /**
     * نوع پیام
     **/
    public $message_type;
    /**
     * خوانده شده
     **/
    public $message_read;
    /**
     * چت
     **/
    public $chat_id;

    public $operator_id;

    public $sender;

    public $user_id;
    /**
     * @var void
     */
    public $created_at;

    public function model(): \model\ChatMessages {
        return new \model\ChatMessages();
    }

    public function buildMessage(string $adminID): \FwHtml\Elements\Tags\Div {
        $senderId = $this->{"$this->sender" . "_id"};
        $datetime = en_to_fa(jdate("Y/m/d", strtotime($this->created_at))) . ' - ' . en_to_fa(jdate("H:i", strtotime($this->created_at)));
        if ($this->sender == 'operator') {
            /** @var \model\Entity\UsersTableEntity $sender */
            $sender = UsersTable::get($senderId);
            if ($sender instanceof UsersTableEntity or true) {
                return HtmlTags::Div('.direct-chat-msg.right')->Content(
                    HtmlTags::Div('.direct-chat-infos.clearfix')->Content(
                        HtmlTags::Span('.direct-chat-name.float-right')->Content(
                            ($sender instanceof UsersTableEntity ? $sender->user_nameToKnow : "اپراتور حذف شده")
                        ),
                        HtmlTags::Span('.direct-chat-timestamp.mr-2')->Content(
                            $datetime
                        ),

                    ),
                    HtmlTags::Img('.direct-chat-img')->Src('/admin/src/images/Users/placeholder.jpg')->Alt('message user image'),
                    HtmlTags::Div('.direct-chat-text' . ($this->message_type == 'image' ? '.img-content' : ""))
                        ->Data_('toggle', 'tooltip')
                        ->Data_('placement', 'left')
                        ->Title($datetime)
                        ->Content(
                            $this->htmlContent()
                        )
                );
            } else {
                return HtmlTags::Div();
            }
        } else {
            /** @var \model\Entity\UsersEntity $sender */
            $sender = Users::get($senderId);
            if ($sender instanceof UsersEntity) {
                return HtmlTags::Div('.direct-chat-msg')->Content(
                    HtmlTags::Div('.direct-chat-infos.clearfix')->Content(
                        HtmlTags::Span('.direct-chat-name.float-left')->Content(
                            $sender->firstname . ' ' . $sender->lastname
                        ),

                        HtmlTags::Span('.direct-chat-timestamp.float-left.ml-2')->Content(
                            $datetime
                        ),
                    ),
                    HtmlTags::Img('.direct-chat-img')->Src($sender->image())->Alt('message user image'),
                    HtmlTags::Div('.direct-chat-text' . ($this->message_type == 'image' ? '.img-content' : ""))
                        ->Data_('toggle', 'tooltip')
                        ->Data_('placement', 'right')
                        ->Title($datetime)
                        ->Content(
                            $this->htmlContent()
                        )
                );
            } else {
                return HtmlTags::Div();
            }
        }
    }

    public function htmlContent(): string {
        $content = json_decode($this->message_content);
        switch ($this->message_type) {
            case "audio":
                return $this->audioContent($content);
            case "file":
                return $this->fileContent($content);
            case "image":
                return $this->imageContent($content);
            case "pay":
                return $this->payContent($content);
            case "text":
            case "card":
                return $this->textPreview($content);
            case "video":
                return $this->videoPreview($content);
            case "order":
                return $this->orderContent($content);
            default:
                return "خطا در پیشنمایش";
        }
    }

    private function audioContent($content): string {
        return "<audio controls><source type='audio/ogg' src='$content->path'></audio><br>پیام صوتی";
    }

    private function fileContent($content): \FwHtml\Elements\Tags\Div {
        return HtmlTags::Div('.d-flex.flex-wrap')->Attrs(['style' => 'width: 20vw'])->Content(
            HtmlTags::Div('.w-25.d-flex.justify-center.align-items-center')->Content(
                HtmlTags::I()->Class(FontAwesome::File())->Attrs(['style' => 'font-size: 50px']),
                HtmlTags::Span()->Content(
                    $content->type
                )->Attrs(['style' => "position: absolute;color: black;right: 20px;top: 30px;font-weight: bold;font-family: sans-serif;"])
            ),
            HtmlTags::Div('.w-75.d-flex.flex-column')->Content(
                HtmlTags::Label($content->name,),
                HtmlTags::Span()->Content($content->size),
                HtmlTags::A('.btn.btn-danger.p-1.mr-auto.ml-0')
                    ->Attrs(['style' => 'width: fit-content'])->Content(
                        HtmlTags::I()->Class(FontAwesome::Download())
                    )->Download($content->name)->Href($content->path),
            )
        );
    }

    private function imageContent($content): string {
        return HtmlTags::Img('.view-image')->Data_('src',$content->path)->Src($content->thumbnail ?? "",)->Attrs(['style' => 'max-width: 15vw']);
    }

    private function payContent($content): string {
        $payed = $content->isPayed;
        return HtmlTags::Table('.table.table-bordered')->Attrs(['style' => $payed ? "background: #00ff9073;" : ""])->Content(
                HtmlTags::Thead()->Content(
                    HtmlTags::Tr()->Content(
                        HtmlTags::Th()->Content("#"),
                        HtmlTags::Th("شرح"),
                        HtmlTags::Th("قیمت"),
                        HtmlTags::Th("تخفیف"),
                    )
                ),
                HtmlTags::Tbody()->Content(
                    function () use ($content) {
                        $output = [];
                        if (is_iterable($content->items)) {
                            foreach ($content->items as $item) {
                                $output[] = HtmlTags::Tr()->Content(
                                    HtmlTags::Td()->Content($item->index . ''),
                                    HtmlTags::Td()->Content($item->name),
                                    HtmlTags::Td()->Content(price_format($item->price)),
                                    HtmlTags::Td()->Content($item->discount . '%'),
                                );
                            }
                        }
                        return implode('', $output);
                    }
                ),
                HtmlTags::Tfoot()->Content(
                    HtmlTags::Tr()->Content(
                        HtmlTags::Td()->Attrs(['colspan' => 2])->Content(
                            "مجموع:"
                        ),
                        HtmlTags::Td()->Attrs(['colspan' => 2])->Content(
                            price_format($content->total)
                        )
                    ),
                    HtmlTags::Tr()->Content(
                        HtmlTags::Td()->Attrs(['colspan' => 2])->Content(
                            "تخفیف:"
                        ),
                        HtmlTags::Td()->Attrs(['colspan' => 2])->Content(
                            price_format($content->totalDiscount) . ' (' . $content->totalDiscountPercent . '%)'
                        )
                    ),
                    HtmlTags::Tr()->Content(
                        HtmlTags::Td()->Attrs(['colspan' => 2])->Content(
                            "پرداخت شده:"
                        ),
                        HtmlTags::Td()->Attrs(['colspan' => 2])->Content(
                            price_format($content->previousPayment)
                        )
                    ),
                    HtmlTags::Tr()->Content(
                        HtmlTags::Td()->Attrs(['colspan' => 2])->Content(
                            "مانده:"
                        ),
                        HtmlTags::Td()->Attrs(['colspan' => 2])->Content(
                            price_format($content->duePayment)
                        )
                    )
                )
            ) . HtmlTags::Span('.mt-1')->Content(
                $payed ? "پرداخت شده" : "در انتظار پرداخت"
            );
    }

    private function textPreview($content): string {
        return $content->text ?? "-";
    }

    private function videoPreview($content): string {
        return "ویدئو";
    }

    private function orderContent($content) {
        return HtmlTags::Table('.table.table-bordered')->Content(
            HtmlTags::Thead()->Content(
                HtmlTags::Tr()->Content(
                    HtmlTags::Th()->Content(
                        "عنوان"
                    ),
                    HtmlTags::Th()->Content(
                        "مقدار"
                    )
                )
            ),
            HtmlTags::Tbody()->Content(
                function () use ($content) {
                    $output = [];
                    /** @var OrdersEntity $order */
                    $order = Orders::get(Chats::get($this->chat_id)->order_id);
                    /** @var ServicesEntity $service */
                    $service = Services::get($order->service_id);
                    $form = json_decode($service->service_form);
                    foreach ($content as $key => $item) {
                        $output[] = HtmlTags::Tr()->Content(
                            HtmlTags::Td()->Content(
                                is_object($form->$key) ? $form->$key->label : $key,
                            ),
                            HtmlTags::Td()->Content(
                                is_file(__SOURCE__ . 'images/Orders/Temp/' . $item) ? HtmlTags::Img('.view-image')->Data_('src',$item)->Src(__IMAGES__ . 'Orders/Temp/' . $item)
                                    ->Attrs(['style' => 'height: 12vh'])
                                    ->Alt($item) : $item,
                            )
                        );
                    }
                    return implode('', $output);
                }
            )
        );
    }

    public function htmlPreview(): string {
        $content = json_decode($this->message_content);
        switch ($this->message_type) {
            case "audio":
                return $this->audioPreview($content);
            case "file":
                return $this->filePreview($content);
            case "image":
                return $this->imagePreview($content);
            case "pay":
                return $this->payPreview($content);
            case "card":
            case "text":
                return $this->textPreview($content);
            case "video":
                return $this->videoPreview($content);
            default:
                return "خطا در پیشنمایش";
        }
    }

    private function audioPreview($content): string {
        return "پیام صوتی";
    }

    private function filePreview($content): string {
        return "فایل $content->type";
    }

    private function imagePreview($content): string {
        return "پیام تصویری";
    }

    private function payPreview($content): string {
        $pay = price_format($content->duePayment);
        return "فاکتور: $pay (قابل پرداخت)";
    }

    public function pushFormat(): array {
        return [
            'type'    => $this->message_type,
            'content' => json_decode($this->message_content, true),
        ];
    }

    public function apiFormat(UsersEntity $user): array {
        return [
            'id'      => $this->chat_message_id,
            'content' => $this->message_content,
            'type'    => $this->message_type,
            'isRead'  => $this->message_read == 1,
            'isMe'    => ($this->sender == 'user' and $this->user_id == $user->user_id),
            'date'    => strtotime($this->created_at),
            'chatId'  => $this->chat_id,
        ];
    }

    public function preview() {
        switch ($this->message_type) {
            case "audio":
                return "صوت";
            case "file":
                return "فایل";
            case "image":
                return "تصویر";
            case "pay":
                return "فاکتور";
            case "text":
                return json_decode($this->message_content)->text;
            case "video":
                return "ویدئو";
            default:
                return "پیام جدید";
        }
    }

    protected function dictionary(): array {
        return [
            'chat_message_id' => 'chat_message_id',
            'message_content' => 'message_content',
            'message_type'    => 'message_type',
            'message_read'    => 'message_read',
            'chat_id'         => 'chat_id'
        ];
    }
}
